;;=========================================================================
;;                            MICRO MOPS
;;
;; An implementation of micro-MOPs in Scheme4.0a.
;; Based on the version described in _Inside Case-Based Reasoning_
;; (Reisbeck and Schank)
;;
;; [2/1/92 Raja Sooriamurthi]
;;=========================================================================

;(require 'lib)
;(load "~/lib/scheme/lib.ss")

; Some macros and functions  from my lib
;------------------------------------------------------------
(extend-syntax (push!)
  [(push! x stk) (set! stk (cons x stk))])

; simple macros for counters
; if called with just the counter
; the increment value is defaulted to 1.
; Incr! and decr! return their updated value.

(extend-syntax (incr!)
  [(incr! c) (symbol? 'c) (incr! c 1)]
  [(incr! c n)
   (and (symbol? 'c) (number? 'n))
   (begin (set! c (+ c n)) c)])

(define snoc
  (lambda (ls x)
    (append ls (list x))))

;------------------------------------------------------------
;; a simple type-checker/error handler

;(extend-syntax (insist)
;  [(insist f) #t]
;  [(insist f p rest ...)
;   (if p
;       (insist f rest ...)
;       (error 'f "failed in ~a" 'p))])

(extend-syntax (insist)
  [(insist f) #t]
  [(insist f (p e0 ...)  rest ...)
   (if (p e0 ...)
       (insist f rest ...)
       (error 'f "~n   Failed in ~a: ~a" '(p e0 ...) `(p ,e0 ...)))]
  [(insist f p rest ...)
   (if p (insist f rest ...)
	(error 'f "~n Failed in ~a: ~a" 'p p))])


; SETF
;--------------------------------------------------------------------------
; Based on the Common Lisp version

(define *fn-list* '())

(extend-syntax (setf getf)
  [(setf x val)
   (symbol? 'x)
   (set! x val)]
  [(setf (getf plist key) val)
   ; Evaluate plist* once
   (let ([plist*  plist])
     (let loop ([pl plist*])
       (cond
         [(null? pl) ; add as 1st element in the plist
          (setf plist (cons key (cons val plist*)))]
         [(eqv? key (car pl)) (set-car! (cdr pl) val)]
         [else (loop (cddr pl))])))]
  [(setf (fn1 a0 a1 ...) val)
   (let ([accessor-setter-pair (assq 'fn1 *fn-list*)])
     (if accessor-setter-pair
         ((cdr accessor-setter-pair) a0 a1 ... val)
         (error 'setf "No matching setter for ~a" 'fn1)))])


; Addf and removef are additions of my own.
; Adds entry to the list of values at (accessor key)

(extend-syntax (addf)
  [(addf (accessor key) entry)
   (setf (accessor key)
         (cons entry (accessor key)))])

; Removes entry from the list of values at (accessor key)

(extend-syntax (removef)
  [(removef (accessor key) entry)
   (setf (accessor key)
         (remq entry (accessor key)))])

(extend-syntax (defsetf)
  [(defsetf fn1 fn2)
   (symbol? 'fn1)
   (push! (cons 'fn1 fn2) *fn-list*)])



; PROPERTY-LISTs
;--------------------------------------------------------------------------
; Property lists are lists of <key> <val> sequences and hence
; are always of even length: ( <key1> <val1> <key2> <val2> ...).
; They are similar to alists except that they are flat and hence
; take up less space. Let a, c, e be keys and b, d, f, their
; associated values then the two representations would be as
; below:
;     plist:  (a b c d e f)
;     alist:  ((a b) (c d) (e f))
; If we consider just a list of symbols a plist would require
; n cons cells where as an alist would require 3n/2 cells where
; n is the total number of symbols (keys and values).
; In general an alist would require k more cells than an equivalent
; plist where k is the # of keys.

; Some additional functions for handling property lists

;;  An accessor for plists

(define getf  
  (lambda (p-list key)
    (let loop ([pl p-list])
      (cond
        [(null? pl) #f]
        [(eqv? key (car pl)) (cadr pl)]
        [else (loop (cddr pl))]))))


; Return a new plist with the 'key' element
; and its associated value removed.

(define remf
  (lambda (p-list key)
    (let loop ([pl p-list])
      (cond
        [(null? pl) '()]
        [(eqv? (car pl) key) (cddr pl)]
        [else (cons (car pl)
                    (cons (cadr pl)
                          (loop (cddr pl))))]))))

; Destructively remove the key.
; The below method is simple but creates garbage.
; Garbage could be avoided by destructively
; modifying the plist (remf! could then be a function).

(extend-syntax (remf!)
  [(remf! pl key)
   (setf pl (remf pl key))])


; TABLES
;--------------------------------------------------------------------------
; Tables are property-lists along with some associated
; fucntions for manipulating them.
; Apart from the macro 'define-table' which is used to create
; tables,  four functions are define for TABLES:
; 1. an accessor (if the key isn't in the table the accessor
;                 returns () (not #f).)
; 2. a setter
; 3. a function to enumerate the keys
; 4. a function to delete the key-value pairs

(extend-syntax (define-table)
  [(define-table accessor vars place)   ; place is an plist
   (with ([key (car 'vars)]
          [setter (string->symbol (format "set-~a!" 'accessor))])
     (begin
       (set! accessor
             (lambda (key)
               ; (or (f ...) '()) seems to be an interesting
               ; way to convert functions returning #f to return '().
               (or (getf place key) '())))
       (set! setter
             (lambda (name val)
               (setf (getf place name) val)))
       (defsetf accessor setter)))])


(define table-keys
  (lambda (table)
    (if (null? table)
        '()
        (cons (car table) (table-keys (cddr table))))))

; ICBR defines delete-key as a function which is incorrect
; as the first key in the table then can't be deleted.

(extend-syntax  (delete-key)
  [(delete-key table key)
   (begin (remf! table key)
          table)])
      


; MOPS                                                    (Listing 3.5 p68)
;--------------------------------------------------------------------------
; MOPS are represented as symbols.
; MOPS are linked to other MOPS via abstraction
; and packaging links.

; *mop-tables* is a table of tables indexed by table names
; Each sub-table is a table of MOP information
; indexed by MOP name
;
; Thus (mop-table 'absts) => 'a table of immediate abstractions'
; and  (mop-absts <mop>)  => 'immediate abstractions of <mop>'
;
; All tables are put in the master table *mop-tables*
; for easier manipulation such as removing a MOP from
; memory.

(setf *mop-tables* '())
(define-table mop-table (table-name) *mop-tables*)

; (mop-absts <mop>) => immediate abstractions of <mop>
; all MOPS exactly one abstraction link above <mop>
; Doesn't include <mop> itself
(define-table mop-absts (mop) (mop-table 'absts))

; (mop-all-absts <mop>) => all the abstractions of <mop>
(define-table mop-all-absts (mop) (mop-table  'all-absts))

; (mop-specs <mop>) => immediate specializations of <mop>
;  [opposite of mop-absts]
(define-table mop-specs (mop) (mop-table 'specs))

; (mop-slots <mop>) => packaging links of <mop>
; an entry is a slot == ( <role> filler> )
(define-table mop-slots (mop) (mop-table 'slots))

; (mop-type <mop>) => either 'instance' or 'mop'
(define-table mop-type (mop) (mop-table 'type))


; MOP predicates                                          (Listing 3.6 p69)
;--------------------------------------------------------------------------

; a MOP is either a number or a symbol stored in the
; mop-type table (as all mops have types)

(define mop?
  (lambda (x)
    (or (number? x)
        (and (symbol? x)
             (not (null? (mop-type x)))))))

; Is it an instance mop?
(define instance-mop?
  (lambda (x)
    (and (mop? x)
         (or (number? x)
             (eqv? (mop-type x) 'instance)))))

; Is it an abstraction mop?
(define abst-mop?
  (lambda (x)
    (and (mop? x)
         (eqv? (mop-type x) 'mop))))

; Is abst an abstraction (not necessarily immediate)
; of spec.  [All numbers are abstractions of each other]

(define abst?
  (lambda (abst spec)
    (or (eqv? abst spec)
        (member abst (mop-all-absts spec)))))

; Is x a specialization of M-pattern ?
; Pattern MOPS are used to hold pattern matching and role
; filling information. ****

(define pattern?
  (lambda (x)
    (abst? 'M-pattern x)))

; Is x a specialization of M-group ?
; Group MOPs are used to hold groups of MOPs like
; the steps in a recipe or the events in a fight.

(define group?
  (lambda (x)
    (abst? 'M-group x)))


; ROLE access functions                                   (Listing 3.7 p71)
;--------------------------------------------------------------------------
; A slot     == ( <role>  <filler> )
;   <role>   == symbol or number representing role name
;   <filler> == a MOP or ()
; **** These functions are similar to the ones we saw
; when manipulating CDs

(define slot-role car)
(define slot-filler cadr)
(define make-slot
  (lambda (role filler) ; filler should be a MOP
    (list role filler)))

; Return the slot with the role in the list
; of slots x

(define role-slot
  (lambda (role x)
    (insist role-slot (or (mop? x) (list? x)))
    (assq role
          (if (mop? x)
              (mop-slots x)
              x))))
 
; Return the filler associated with the role
; in the list of slots x

(define role-filler
  (lambda (role x)
    (let ([slot (role-slot role x)])
      (and slot (slot-filler slot)))))

; Add the slot (<role> <filler>) to the slots of MOP.
; It's an error if a slot with <role> already exists.

(define add-role-filler
  (lambda (role mop filler)
    (insist add-role-filler
            (mop? mop)
            (not (role-filler role mop)))
    (printf "~s: ~s <= ~s~n" mop role filler)
    (addf (mop-slots mop) (make-slot role filler))
    filler))


; Basic MOP abstraction functions                         (Listing 3.8 p72)
;--------------------------------------------------------------------------

; (link-abst <mop1> <mop2>) makes <mop2> an immediate abstraction
; of <mop1>.  If the mops are already linked nothing happens.
; It also prevents linking a mop to an instance, itself or an
; abstraction of itself.

(define link-abst
  (lambda (spec abst)
    (insist link-abst (abst-mop? abst) (mop? spec)
            (not (abst? spec abst)))
    (when (not (abst? abst spec)) ; when not already linked
      (addf (mop-absts spec) abst)
      (addf (mop-specs abst) spec)
      (redo-all-absts spec))
    spec))


; (unlink-abst <mop1> <mop2>) removes the abstraction link
; between the two mops (if any), updates the abstraction heirarchy
; and returns <mop1>

(define unlink-abst
  (lambda (spec abst)
    (when (abst? abst spec)
      (removef (mop-absts spec) abst)
      (removef (mop-specs abst) spec)
      (redo-all-absts spec))
    spec))

; (redo-all-absts <mop>) re-calculates all the abstractions
; for <mop> and then recursively calls itself to calculate
; all the abstractions for the specializations of <mop>

(define redo-all-absts
  (lambda (mop)
    (setf (mop-all-absts mop) (calc-all-absts mop))
    (for-each redo-all-absts (mop-specs mop))))


; (calc-all-absts <mop>) calculates a list of all the abstractions
; for <mop>.  Its assumed that all the abstractions of the immediate
; abstractions of <mop> are correct. [redo-all-absts works its way down
; because of this.]  Using mop-all-absts all the abstractions of the 
; immediate abstractions of <mop> are made into a list with <mop>
; in the front and then the duplicates are removed retaining the
; last one.  This effectivley flattens the 'mop graph' by performing
; a topological sort such that no abstraction occurs in the list
; before any of its specializations.

(define calc-all-absts
  (lambda (mop)
    (remove-duplicates
     (cons mop
           (apply append
                  (map (lambda (m)
                          (mop-all-absts m)) 
                       (mop-absts mop)))))))
     
; Remove duplicate entries in l
; retaining the last one.

(define remove-duplicates
  (lambda (l)
    (cond
      [(null? l) '()]
      [(memq (car l) (cdr l))
       ; then discard the car
       (remove-duplicates (cdr l))]
      [else (cons (car l) (remove-duplicates (cdr l)))])))
  

; MOP construction functions                              (Listing 3.9 p73)
;--------------------------------------------------------------------------

; (new-mop <name> <mop list> <type> <slot list>) makes a mop named <name>
; of type <type> have immediate abstractions <mop list> and slots
; <slot list>.
;    <type> can be mop, instance or #f; if #f a default type is
;    calculated.
;    <name> must be a symbol or #f; if #f a default name
;    is calculated.

(define new-mop
  (lambda (name absts type slots)
    (insist new-mop (or (symbol? name) (not name))
            (andmap mop? absts))
    (let* ([type* (or type (calc-type absts slots))]
           [name* (or name (spec-name absts type*))])
      (setf (mop-type name*) type*)
      (when (not (null? slots)) (setf (mop-slots name*) slots))
      (for-each
        (lambda (abst)
          (link-abst name* abst))
        absts)
      name*)))

      
; (calc-type <mop-list> <slot-list>) calculates the default type
; a mop with the given abstractions and slots.  The default is:
;       Instance: if (i)  The mop is not a specialization
;                         of a pattern mop.
;                    (ii) Has slots and all the slot fillers
;                         are instances
;       otherwise:
;         its a MOP.
; **** actually the code below does the 'complement' of the
; **** above description; maybe it should be changed to the
; **** '*' version below.

(define calc-type
  (lambda (absts slots)
    (cond
      ; is it a specialization of a pattern mop?
      [(ormap pattern? absts) 'mop]
      [(null? slots) 'mop]
      ; is any slot filler not an instances?
      [(ormap (lambda (slot)
                 (not (instance-mop? (slot-filler slot))))
               slots)
       'mop]
      [else 'instance])))

(define calc-type*
  (lambda (absts slots)
    (cond
      ; is not  a specialization of a pattern mop
      [(andmap (lambda (abst) (not (pattern? abst))) absts) 'instance]
      ; all slot fillers  instances
      [(and (not (null? sots))
	    (andmap (lambda (slot)
                 (instance-mop? (slot-filler slot)))
               slots))
       'instance]
      [else 'mop])))

; (spec-name <mop-list> <type>) builds a name of the form mop.integer
; (or I-mop.integer for instance), where mop is the first mop in the
; <mop-list>.

(define spec-name
  (lambda (absts type)
    (gentemp (format (if (eqv? type 'mop) "~a" "i-~a") (car absts)))))


(define gentemp
  (let ([count 0])
    (case-lambda
      [() (set! count 0)]
      [(str) (let ([fstr (string-append str ".~a")])
	       (begin
		 (incr! count)
		 (string->symbol (format fstr count))))])))


; Memory management functions                            (Listing 3.10 p74)
;--------------------------------------------------------------------------

; clear-memory removes all mops from memory and re-installs
; the root mop M-ROOT.  All mops in memory are attached
; somewhere in memory under M-root

(define clear-memory
  (lambda ()
    (setf *mop-tables* '())
    (gentemp)  ; reset the gentemp counter
    (new-mop 'M-root '() 'Mop '())
    (setf (mop-all-absts 'M-root)
          (calc-all-absts 'M-root))
    'M-root))

; all-mops returns a list of all the mops in memory

(define all-mops
  (lambda ()
    (table-keys (mop-table 'type))))

; (remove-mop <name>) removes the mop named <name> by
; unlinking it from all its abstractions and then deleting it from
; all the tables.

(define remove-mop
  (lambda (name)
    (for-each
      (lambda (abst)
        (unlink-abst name abst))
      (mop-absts name))
    (for-each
      (lambda (table-name)
        (setf (mop-table table-name)
              (delete-key (mop-table table-name) name)))
      (table-keys *mop-tables*))))


; Extended role-access functions                         (Listing 3.11 p75)
;--------------------------------------------------------------------------

; (inherit-filler <role> <mop>) returns the first filler of <role>
; found in either <mop> or one of its abstractions

(define inherit-filler
  (lambda (role mop)
    (ormap
      (lambda (abst)
        (role-filler role abst))
      (mop-all-absts mop))))
      


; (get-filler <role> <mop>) returns the filler for <role> in <mop>
; The filler can either be an inherited value or a calculated value.
; If a filler has to be calculated , a slot for the role and the
; filler is added to the <mop>.  A filler is calculated when <role>
; inherits a pattern <mop> with a calculation function. A calculation
; function is stored in the CALC-FN role of a pattern mop and should
; be defined with two parameters to hold the inherited filler and the
; <mop>.  An important type of calculation function is the adaptation
; function which calculates the filler of a role by adapting the filler
; of the same role in some sibling instance.


(define get-filler
  (lambda (role mop)
     (or (role-filler role mop); if mop has it return that
        (let ([filler (inherit-filler role mop)])
	   (and filler
		 (or (and (instance-mop? filler) filler)
		    (and (abst? 'M-function filler) filler)
		    (let ([fn (get-filler 'calc-fn filler)])
		       (and fn
			     (let ([new-filler (funcall fn filler mop)])
				(and new-filler
				      (add-role-filler role mop
					 new-filler)))))))))))

(define funcall
  (lambda (fn . args)
    (if (symbol? fn)            ; an eval is needed here
	(apply (eval fn) args)  ; as a result of the way we 
	(apply fn args))))      ; define our MOPs and how CL funcall works.


; (PATH-FILLER <path> <mop>) returns the filler for path
; in <mop>.  A <path> is a set of roles and path-filler follows that
; list in order using get-filler.
; eg. (path-filler '(explanation cause) 'M) first gets  the
;     explanation filler of M and if that succeeds, gets the
;     cause filler of the result.
    
(define path-filler
  (lambda (path mop)
    (if (null? path) 
        mop
        (let ([filler (get-filler (car path) mop)])
              (and filler
                   (path-filler (cdr path) filler))))))

; an alternative definition --- not so elegant

(define path-filler*
  (lambda (path mop)
    (and (andmap
           (lambda (role)
             (let ([filler (get-filler role mop)])
               (and (set! mop filler) filler)))
           path)
         mop)))


; Extended MOP Abstraction predicates                    (Listing 3.12 p77)
;--------------------------------------------------------------------------

; (SLOTS-ABST? <mop> <slot-source>) returns true if every slot
; in <mop> is 'satisfied' by the corresponding slot in <slot-source>.
; <slot-source> is usually a mop though it can be a list of slots.
; The slot fillers in <mop> are treated as constraints on the slot-fillers
; in <slot-source>.  <mop> must be an abstraction and have at least
; one slot to be satisfied.

(define slots-abst?
  (lambda (mop slots)
    (and (abst-mop? mop)
         (not (null? (mop-slots mop)))
         (andmap
           (lambda (slot)
             (satisfied? (slot-filler slot)
                         (get-filler (slot-role slot) slots)
                         slots))
           (mop-slots mop)))))

; (SATISFIED? <constraint> <filler> <slot-source>) returns true if
; <filler> satisfies the conditions specified by <constraint>.

(define satisfied?
  (lambda (constraint filler slots)
    (cond
      [(or (not constraint) (null? constraint))]
      [(pattern? constraint)
       (funcall (inherit-filler 'abst-fn constraint)
                constraint filler slots)]
      [(abst? constraint filler)]
      [(instance-mop? constraint) (not filler)]
      [filler (slots-abst? constraint filler)]
      [else #f])))
      


; MOP equality functions                                 (Listing 3.14 p78)
;--------------------------------------------------------------------------

; (MOP-INCLUDES? <mop1> <mop2>) returns <mop1> if the two mops have
; the same type and every slot in <mop2> is also in <mop1>.

(define mop-includes?
  (lambda (mop1 mop2)
    (and (eqv? (mop-type mop1) (mop-type mop2))
         (andmap
           (lambda (slot)
             (eqv? (slot-filler slot)
                   (get-filler (slot-role slot) mop1)))
           (mop-slots mop2))
	 mop1)))

(define mop-equal?
  (lambda (mop1 mop2)
    (and (mop-includes? mop1 mop2)
         (mop-includes? mop2 mop1))))

; (GET-TWIN <mop>) returns the first mop in memory that it can find
; that is equal to mop, if any.  It goes one level up in the
; heirarchy with mop-absts and then comes down with mop-specs.

(define get-twin
  (lambda (mop)
    (ormap
      (lambda (abst)
        (ormap
          (lambda (spec)
            (and (not (eqv? spec mop))
;                (mop-equal? spec mop)   === Bug 
		 (mop-includes? spec mop)
		 (or (not (group? mop))
		     (mop-includes? mop spec))
		 spec))
	    (mop-specs abst)))
      (mop-absts mop))))
    


; Refining functions                                     (Listing 3.14 p78)
;--------------------------------------------------------------------------

; (REFINE-INSTANCE <instance>) takes each abstraction of <instance>
; and tries to replace it with one or more specializations of the
; abstraction.  It repeats this process until all the abstractions
; of <instance> are as specialized as possible.

(define refine-instance
  (lambda (instance)
    (ormap   ;  **** should this be for-each?
      (lambda (abst)
        (if (mops-abst? (mop-specs abst) instance)
            (begin
              (unlink-abst instance abst)
              (refine-instance instance))
            #f))
      (mop-absts instance))))

; **** the above code  seems equivalent to the below
; **** am not sure and will have to check it out

(define refine-instance*
  (lambda (instance)
    (andmap
      (lambda (abst)
        (if (mops-abst? (mop-specs abst) instance)
            (begin
              (unlink-abst instance abst)
              (refince-instance instance))
            #f))
      (mop-absts instance))))


; (MOPS-ABST? <mop-list> <instance>) looks at each mop in <mop-list>.
; If the mop can abstract <instance>, a link from <instance> to
; the mop is made.  True is returned if at least one such mop
; is found.

(define mops-abst?
  (lambda (mops instance)
    (let ([flag #f])
      (for-each
        (lambda (mop)
          (when (slots-abst? mop instance)
            (link-abst instance mop)
            (set! flag #t)))
        mops)
      flag)))

     
; Instance installation functions                        (Listing 3.15 p79)
;--------------------------------------------------------------------------

; (INSTALL-INSTANCE <instance>) puts an instance in memory.  First it
; refines the instance's abstractions, then it checks if any instance
; in memory includes this one.  If so, <instance> is removed and the
; old instance is returned.  If not, illegal abstractions are
; unlinked.  If any abstractions are left, <instance> is returned,
; otherwise <instance> is removed and false is returned.

(define install-instance
  (lambda (instance)
    (refine-instance instance)
    (let ([twin (get-twin instance)])
      (cond [twin (remove-mop instance) twin]
            [(has-legal-absts? instance) instance]
            [else  (remove-mop instance) #f]))))


; (HAS-LEGAL-ABSTS? <instance>)  unlinks every immediate abstraction
; of <instance> that is not a legal place to put <instance>, and
; returns true if any immediate abstractions are left.

(define has-legal-absts?
  (lambda (instance)
    (for-each
      (lambda (abst)
        (when (not (legal-abst? abst instance))
          (unlink-abst instance abst)))
      (mop-absts instance))
      (not (null? (mop-absts instance)))))
    

; (LEGAL-ABST? <mop> <instance>) returns true if <mop> is a legal
; place to put <instance>, i.e., <mop> is not slotless and does not
; have abstractions below it.

(define legal-abst?
  (lambda (abst instance)
    (and (not (null? (mop-slots abst)))
         (andmap instance-mop? (mop-specs abst)))))


; Abstraction installation functions                     (Listing 3.16 p80)
;--------------------------------------------------------------------------

; (INSTALL-ABSTRACTION <mop>) installs a new abstraction.  If an
; identical mop  is already in memory, it is returned.  Otherwise, any
; instances of mop's abstractions that can go under <mop> are put
; there and <mop> is returned.

(define install-abstraction
  (lambda (mop)
    (let ([twin (get-twin mop)])
      (cond [twin (remove-mop mop) twin]
	    [else (reindex-siblings mop)]))))

; (REINDEX-SIBLINGS <mop>) finds all instances that are immediate
; specializations of immediate abstractions of <mop>.  The instances
; are unlinked from their abstractions and relinked to <mop> and <mop>
; is returned.

(define reindex-siblings
  (lambda (mop)
    (for-each
      (lambda (abst)
        (for-each
          (lambda (spec)
            (when (and (instance-mop? spec)
                       (slots-abst? mop spec))
              (unlink-abst spec abst)
              (link-abst spec mop)))
          (mop-specs abst)))
      (mop-absts mop))
    mop))
            

; Top-level memory update functions                      (Listing 3.17 p81)
;--------------------------------------------------------------------------

; (SLOTS->MOP  <slot-list> <mop-list> <must-work>)  returns a mop with
; the slots in <slot-list> and the abstractions in <mop-list>.  If the
; first element in <slot-list> is an atom it is the type of mop
; desired -- mop or instance.  If <slot-list> is empty and <mop-list>
; contains only one mop, that mop is returned immediately.  Otherwise
; a mop is constructed and installed as an instance mop or
; abstraction as appropriate.  Installation may return an existing mop
; in memory or the mop just constructed.  If <must-work> is true, an
; error is signaled if a mop can't be installed for some reason.  If
; <must-work> is false, then #f is returned when installation fails.


(define slots->mop
  (lambda (slots absts must-work)
    (insist slots->mop (not (null? absts)) (andmap mop? absts))
    (or (and (null? slots) (null? (cdr absts)) (car absts))
        (let* ([type (and (not (null? slots))
                          (atom? (car slots)) (car slots))]
               [slots (if type (cdr slots) slots)]
               [mop (new-mop #f absts type slots)]
               [result (if (instance-mop? mop) 
			   (install-instance mop)
			   (install-abstraction mop))])
          (insist slots->mop (or result (not must-work)))
          result))))



; Form to MOP functions                                  (Listing 3.18 p82)
;--------------------------------------------------------------------------

; (DEFMOP <name> <mop-list> <type> <slot-form1> <slot-form2> ...)
; <type> is an optional argument specifying the type of the mop being
; defined --- if present it should either be MOP or INSTANCE.  If
; omited it is calculated by CALC-TYPES.
;
; DEFMOP defines a mop named <name> with the immediate abstractions
; <mop-list>, type <type> and slots as specified by the slot-forms
; <slot-formi>.  A slot form is a list containing a role name, a MOP,
; an optional MOP type and zero or more slot forms.;
;     slot-form ::= ( {<role-name>  <mop> [<mop-name>]}+ )

(extend-syntax (defmop)
  [(defmop name absts args ...)
   (defmop-fn 'name 'absts 'args ...)])

(define defmop-fn
  (lambda (name absts . args)
   (let* ([type (and (not (null? args))
                    (atom? (car args))
                    (car args))]
          [slot-forms (if type (cdr args) args)])
     (new-mop name absts type (forms->slots slot-forms)))))


; (FORMS->SLOTS <slot-form-list> <mop>)  converts a list of slot forms
; into a list of slots.  A slot is a list of the form 
; (<role> <filler-mop>).  A slot-form may have this form or it may
; have one or more slot-forms following <filler-mop>.  In this case
; the slot forms are converted to slots recursively and the result is
; turned into a specializaton of <filler-mop> using slots->mop.  If
; the first element of <slot-form-list> is an atom, it is kept for
; later use as a mop type by slots->mop.


(define forms->slots
  (lambda (slot-forms)
    (map
      (lambda (slot-form)
        (if (atom? slot-form)
            slot-form
            (make-slot (slot-role slot-form)
                       (let ([abst (cadr slot-form)])
                         (insist forms->slots
                                 (atom? abst))
                         (and abst
                              (slots->mop
                               (forms->slots (cddr slot-form))
                               (list abst)
                               #t))))))
      slot-forms)))

; Group MOP functions                                    (Listing 3.19 p83)
;--------------------------------------------------------------------------

; A group MOP represents a set of MOPs such as the steps in a recipe.
; If n is the size of the group then the slots of the group are
; (1 <mop1>), (2 <mop2>) ... (n <mopn>)

; (GROUP-SIZE <group>) returns the size of the group

(define group-size
  (lambda (x)
    (and (group? x)
         (length (mop-slots x)))))

; (GROUP->LIST <group>) returns a list of the members of the group.

(define group->list
  (lambda (group)
    (and group ; **** (null? group)
         (insist group->list (group? group))
         ; 'sort' could probably be used here to get
         ; the fillers in the right order.
         (let loop
             ([indices (make-m-n 1 (group-size group))])
	   (if (null? indices)
	       '()
	       (let ([filler (role-filler (car indices) group)])
		 (if filler
		     (cons filler (loop (cdr indices)))
		     (loop (cdr indices)))))))))

; (LIST->GROUP <list>)  returns a group MOP with members from <list>.
; If <list> is empty the special empty group is returned.

(define list->group
  (lambda (lst)
    (if (null? lst)
        'I-M-Empty-Group
        (slots->mop
         (let loop ([l lst] [i 1])
           (if (null? l)
               '()
               (cons (make-slot i (car l))
                     (loop (cdr l) (add1 i)))))
         '(M-Group)
         #t))))



; (make-m-n <int1> <int2>) returns a list of integers from <int1> to
; <int2> inclusive.

(define make-m-n
  (lambda (m n)
    (let ([next (if (< m n) add1 sub1)])
      (let loop ([m m])
        (if (= m n)
            (list m)
            (cons m (loop (next m))))))))
          


; MOP display functions                                  (Listing 3.20 p85)
;--------------------------------------------------------------------------

; (DAH  <mop>) prints all the specializations under <mop>.
; 'Display Abstraction hierarchy'

(define dah
  (case-lambda
    [() (dah 'm-root)]
    [(mop)
     (pretty-print
      (tree->list mop specs->list '()))]))

; (DPH <mop>) prints all the slots of <mop>, the slots of the fillers
; of the slots of <mop> and so on.  'Display Packaging hierarchy'

(define dph
  (lambda (mop)
    (pretty-print
     (tree->list mop slots->forms '()))))

; (SPECS->LIST <mop>  <mop-list>)  returns a list starting with <mop>
; followed by the specialization tree under each specializations of
; <mop>

(define specs->list
  (lambda (mop visited)
    (map (lambda (spec)
           (tree->list spec specs->list visited))
         (mop-specs mop))))

; (SLOTS->FORMS <mop> <mop-list>) converts the slots in <mop> into a
; list of slot-forms by expanding out the slots in each slot filler.

(define slots->forms
  (lambda (mop visited)
    (map (lambda (slot)
           (cons (slot-role slot)
                 (mop->form (slot-filler slot)
                            visited)))
         (mop-slots mop))))

; (MOP->FORM <mop> <mop-list>) returns a list starting with <mop>
; followed by slots of <mop> converted to slot-forms

(define mop->form
  (lambda (mop visited)
    (tree->list mop slots->forms visited)))
     

; (TREE->LIST <mop> <function> <mop-list>) returns a list with <mop>
; followed by the elements of the list returned by calling <function>
; with <mop> and <mop-list> updated to include <mop>.  If <mop> is
; already in <mop-list> just a list with <mop> is returned.

(define tree->list
  (lambda (mop fn visited)
    (if (memq mop visited)
        (list mop)
        (cons mop
              (funcall fn mop (cons mop visited))))))


; Basic Abstraction and Calculation functions            (Listing 3.22 p87)
;--------------------------------------------------------------------------

; All abstraction functions take three arguments

; (CONSTRAINT-FN <constraint> <filler> <slot-source>) imposes no
; constrain on <filler> so it always returns true

(define constraint-fn
  (lambda (constraint filler slots)
    #t))

; (NOT-CONSTRAINT <constraint> <filler> <slot-source>) is true if
; <filler> does not satisfy the contstraint found in the object slot
; of <contstraint>

(define not-constraint
  (lambda (constraint filler slots)
    (insist not-constraint (not (null? filler)))
    (not (satisfied? (get-filler 'object constraint)
                     filler slots))))

; (GET-SIBLING <pattern> <mop>) finds a sibling of <mop>)
                               
(define get-sibling
  (lambda (pattern mop)
    (ormap
      (lambda (abst)
        (ormap
          (lambda (spec)
            (if (and (instance-mop? spec)
                       (not (eqv? spec mop))
                       (not (abst? 'M-failed-solution spec)))
                spec
                #f))
          (mop-specs abst)))
      (mop-absts mop))))

;==========================================================================
              
