;;=========================================================================
;;                            MICRO CHEF
;;
;; An implementation of micro-MOPs in Scheme4.0a.
;; Based on the version described in _Inside Case-Based Reasoning_
;; (Reisbeck and Schank)
;;
;; [2/1/92 Raja Sooriamurthi] 
;;
;; 4/7/94 Added a check to run-chef to insist that input slots had
;;        mops in them  
;;
;;=========================================================================

;;;(require "microMOPs")

; Top level micro CHEF function                          (Listing 7.1 p214)
;--------------------------------------------------------------------------
; (CHEF <slot-list>) finds or creates a case under M-recipe with the
; given slots and returns it.  The call to get-filler forces the
; calculation of the role steps.  (dph (role-filler 'steps <recipe>))
; will print a readable version of the steps in the recipe returned by
; CHEF.

(define chef
  (lambda (slots)
    (let  ([instance (slots->mop slots '(M-recipe) #t)])
      (and (get-filler 'steps instance)
	   instance))))


; Substitution functions                                 (Listing 7.5 p218)
;--------------------------------------------------------------------------

; MOP-SUBST is a general substitution function for creating a new MOP
; from an old one by replacing some element of the old MOP with a new
; element.  This susbstitution may change a MOP into a 'non-MOP'.
; There might be a redundant step or the abstractions might have to be
; changed.  For instance when changing chicken to beef in a recipe the
; step of boning the chicken has to be removed (as beef isn't bony).
; Similarly the chopping of brocolli might be classified as a hard
; step and when green peppers are substituted the classification might
; have to change (an easy step).

; (mop-subst <mop1> <mop2> <mop3>) returns a new mop like <mop3> where
; all occurences of <mop2> have been replaced with <mop1>.  If the
; modified mop doesn't fit in memory the result will be #f.  If the
; slots in a group have been changed list->group is used to renumber
; the slots correctly.

(define mop-subst
  (lambda (new old mop)
    (cond
      [(eqv? mop old) new]
      [(null? (mop-slots mop)) mop]
      [else (let* ([slots (mop-slots mop)]
		   [new-slots (slots-subst new old slots)])
	      (cond
		[(equal? new-slots slots) mop]
		[(group? mop)
		 (list->group (map slot-filler new-slots))]
		[else (raise-abst (forms->slots new-slots) mop)]))])))


; (slots-subst <mop1> <mop2> <slot-list>) returns a new list of slots
; with <mop1> replacing <mop2> in each filler of <slot-list>.  Slots
; with #f fillers after substitution are not included in the result.

(define slots-subst
  (lambda (new old slots)
    (let loop
	([slots slots])
      (if (null? slots)
	  '()
	  (let ([filler (mop-subst new old (slot-filler (car slots)))])
	    (if filler
		; filter out #f fillers
		(cons (make-slot (slot-role (car slots)) filler)
		      (loop (cdr slots)))
		(loop (cdr slots))))))))


; (raise-abst <slot-list> <mop>) searches the abstractions of <mop>
; until it finds one where it can put a mop with the slots in
; <slot-list>.  It returns the slot-list or #f if there is no
; abstraction that can hold <slot-list>.

(define raise-abst
  (lambda (slots mop)
    (let ([abst (ormap
		  (lambda (abst)
		    (if (slots-abst? abst slots)
			abst
			#f))
		  (mop-all-absts mop))])
      (and abst
	   (slots->mop slots (list abst) #f)))))
		 

; Dynamic MOP making functions                           (Listing 7.6 p219)
;--------------------------------------------------------------------------

; Functions and MOPs for building new MOPs.
; (make-mop <pattern> <mop>) where <pattern> provides the template and
; <mop> provides some of the pieces.  The mop role of pattern says
; what kind of MOP to build (eg., M-cook-step) and the other slots in
; <pattern> say what slots the new MOP should have.  If a slot in
; <pattern> is filled with the name of a role,  then the filler of
; that role in <mop> is used.

(define make-mop
  (lambda (pattern mop)
    (let ([slots (let loop
		     ([slots (mop-slots pattern)])
		   (cond
		     [(null? slots) '()]
		     [(not (eqv? (slot-role (car slots)) 'mop))
		      (cons (car slots) (loop (cdr slots)))]
		     [else (loop (cdr slots))]))])
      (slots->mop (replace-slots slots mop)
		  (list (get-filler 'mop pattern))
		  #t))))

; (replace-slots <slot-list> <mop>) returns a slot-list with every
; filler in a role name replaced by the filler of that role in <mop>.
; In order for replace-slots to know that something is a role name,
; it must have been defined to be an instance of M-role.

(define replace-slots
  (lambda (slots mop)
    (map
      (lambda (slot)
	`(,(slot-role slot)
	  ,(let ([filler (slot-filler slot)])
	     (if (abst? 'M-role filler)
		 (role-filler filler mop)
		 filler))))
      slots)))


; to facilitate loading these mops are in microCHEF.mops
;    (defmop make-mop (m-function))
;    (defmop m-make-mop (m-pattern)
;        (calc-fn make-mop)) ;;; (role m-role))


; Calculating preconditions                              (Listing 7.7 p220)
;--------------------------------------------------------------------------

; The previous functions can replace ingredients in a recipe and
; remove unnecessary steps (such as boning beef).  The below function
; adds steps.  If we replace green beans with brocolli, we will need
; to add a 'chop brocolli' step,  because green beans did not need
; chopping in the original recipe.  Such additional steps are called
; preconditions, and are handled by the below functions nad MOPs

; (get-precons <ingred>) returns a mop returns a mop similar in form
; to M-recipe-steps containing those steps that <ingred> requires.

(define get-precons
  (lambda (ingred)
    (printf "--------------------~n")
    (printf "Getting preconditions for ~a~n" ingred)
    (and ingred
	 (slots->mop `((ingred ,ingred)) '(M-precons) #f))))

; to facilitate loading these mops are in microCHEF.mops
;   (defmop ingred (m-role) instance)
;   (defmop m-precons (m-root))
;
; bony ingredients need a boning step
;
;(defmop m-bony-ingred-precons (m-precons)
;    (ingred m-bony-meat)
;    (bone-steps m-make-mop (mop m-bone-step)
;                (object ingred)))
;
; large ingredients need a chopping step
;
;(defmop m-large-ingred-precons (m-precons)
;    (ingred m-large-ingred)
;    (chop-steps m-make-mop (mop m-chop-step)
;                (object ingred)))


; Precondition functions                                 (Listing 7.8 p221)
;--------------------------------------------------------------------------
; General functions for manipulating groups

; (group-member? <mop> <group>) returns true if <mop> is a member of
; <group>.
; Group-member? revised and object-slot-member added on 2/14/93 to
; allow recognition of membership within nested groups. **********

(define group-member?
  (lambda (mop group)
    (and (group? group)
	 (ormap
	   (lambda (slot)
	     (eqv? (slot-filler slot) mop))
	   (mop-slots group)))))

;	    (lambda (slot)
;	       (or (eqv? (slot-filler slot) mop)
;		  (object-slot-member? (get-filler 'object mop)
;		     (get-filler 'object (slot-filler slot)))))




(define object-slot-member?
   (lambda (fil1 fil2)
      (and fil1 fil2
         (or (eqv? fil1 fil2)
            (and (group? fil2)
                 (ormap
                    (lambda (slot)
                       (eqv? fil1 (slot-filler slot)))
                    (mop-slots fil2)))))))


; (group-splice <mop-list> <mop> <group>) returns a new group MOP with
; all the element of <group> except that <mop> is replaced with the
; elements of <mop-list>.  Not that a '() mop-list returns a group
; with mop removed.

(define group-splice
  (lambda (new old group)
    (let ([new-l
	   (let loop
	       ([lst (group->list group)])
	     (cond
	       [(null? lst) '()]
	       [(eqv? old (car lst)) (append new (loop (cdr lst)))]
	       [else (cons (car lst) (loop (cdr lst)))]))])
      (list->group new-l))))


; (group-insert <mop> <group>) returns a new group MOP with all the
; elements of <group> plus <mop> added at the end.

(define group-insert
  (lambda (mop group)
    (if (or (not mop)
	    (group-member? mop group))
	group
	(list->group
	 (snoc (group->list group) mop)))))



; Precondition merging functions                         (Listing 7.9 p222)
;--------------------------------------------------------------------------
; Functions that merge the steps in the MOP returned by get-precons
; with the steps in an old recipe.

; (add-precons <precon-mop> <slot-list>)  merges the slots of
; precon-map with <slot-list>.

(define add-precons
  (lambda (precon-mop steps-slots)
    (cond
      [(not precon-mop) steps-slots]
      [else (printf "Adding preconditions to recipe~n")
	    (map
	      (lambda (slot)
		(merge-step precon-mop slot))
	      steps-slots)])))

; (merge-step <precon-mop> <slot>) adds the filler of slot to the
; group MOP in the corresponding slot in <precon-mop>,  unless it is
; already there.

(define merge-step
  (lambda (precon-mop slot)
    (let* ([role (slot-role slot)]
	   [old-filler (slot-filler slot)]
	   [new-filler (group-insert (get-filler role precon-mop)
				     old-filler)])
      (if (eqv? new-filler old-filler)
	  slot
	  (make-slot role new-filler)))))

; Adaptation functions                                  (Listing 7.10 p223)
;--------------------------------------------------------------------------

; Functions that adapt an old recipe to use new ingredients.  The
; corresponding ingredient roles in the old and new recipes are
; compared.  The following possibilities can hold with each role:
;   - The old and new recipes have the same ingredient.  The old
;     recipe needs no changes.
;   - The new recipe has #f:  it inherits the old ingredient and the
;     old recipe needs no changes.
;   - The old recipe has #f:  the preconditions for the new ingredient
;     have to be added to the old recipe.
;   - The old and new recipes have different ingredients:  the old
;     ingredient has to be replaced with the new one, and the
;     preconditions for the new ingredients have to be added.

; (adapt-steps <pattern> <mop>) fills the steps slot of <mop> by
; adapting the steps slot of a sibling instance of <mop>.  adapt-steps
; will return #f immediately if the set of steps becomes '().

(define adapt-steps
  (lambda (pattern mop)
    (let ([recipe (get-filler 'old mop)])
      (printf "--------------------~n")
      (printf "Adapting the steps in ~a~n" recipe)
      (let ([steps (mop-slots (get-filler 'steps recipe))])
	(and
	 (andmap
	   (lambda (ingred-role)
	     ; ****
	     (set! steps
		   (subst-ingred
		    (get-filler ingred-role mop)
		    (get-filler ingred-role recipe)
		    steps))
	     steps)
	   '(meat vege vege2))
	 (slots->mop steps '(M-recipe-steps) #f))))))


; (subst-ingred <ingred1> <ingred2> <pattern> <slot-list>)  first
; replaces <ingred2> with <ingred1> in the fillers of the slots in
; <slot-list> and then adds any preconditions of <ingred1> not in the
; modified list.  The substitution process removes steps such as
; boning a non-bony meat.  The modified list of slots is returned.

(define subst-ingred
  (lambda (new-ingred old-ingred steps)
    (cond
					; ****
      [(or (not new-ingred)
	   (eqv? new-ingred old-ingred))
       steps]
      [else (let* ([precon-mop (get-precons new-ingred)]
		   [steps-slots
		    (if (not old-ingred)
			steps
			(begin
			  (printf "Substituting ~a for ~a~n"
				  new-ingred old-ingred)
			  (slots-subst new-ingred old-ingred steps)))])
	      (add-precons precon-mop steps-slots))])))
	     
	       
	       
; Top-level repair function                             (Listing 7.14 p226)
;--------------------------------------------------------------------------

; Chef-repair is similar to the top-level function for recipe
; generation.
;
; (chef-repair <slot-list>) finds or creates a case under M-repair
; with the given slots and returns it.  First the solution is marked
; as a failure, to avoid using it in future adaptations.

(define chef-repair
  (lambda (slots)
    (link-abst (role-filler 'solution slots) 'M-failed-solution)
    (let ([instance (slots->mop slots '(M-repair) #t)])
      (and (get-filler 'repaired-solution instance)
	   instance))))


; Role resetting function                               (Listing 7.17 p228)
;--------------------------------------------------------------------------

; A general function for creating a new mop from an old one by
; changing the filler of a role.
;
; (reset-role-filler <role> <mop> <filler>) returns a new mop with the
; slots of <mop> changed to included ( <role> <filler> ).

(define reset-role-filler
  (lambda (role mop new-value)
    (let ([old-value (role-filler role mop)])
      (insist reset-role-filler old-value) ; ****
      (if (eqv? old-value new-value)
	  mop
	  (raise-abst
	   (map
	     (lambda (slot)
	       (if (eqv? (slot-role slot) role)
		   (make-slot role new-value)
		   slot))
	     (mop-slots mop))
	   mop)))))


; Role calculation functions                            (Listing 7.18 p229)
;--------------------------------------------------------------------------

;  Functions used to generate the repair for the soggy brocoli recipe
;  and apply that repair to create a new recipe.

; (split-step <pattern> <mop>) returns a group of steps that should be
; used in place of the step that caused the failure.  It assumed that
; the step (found in the cause slot of the explanation) has a group
; filling its object slot.  split-step generates two steps.  The first
; one applies the step's action to just the ingredients that ended up
; in a failure state, and the second one applies the action to the
; remaining ingredients.  In the exaple, split-step will return a
; group MOP with a step that stir fries the broccoli step, and a step
; that stir fries the beef and other ingredients.

(define split-step
  (lambda (pattern mop)
    (let ([step (path-filler '(explanation cause) mop)]
	  [object (path-filler '(explanation failure object) mop)])
      (let ([step-object (role-filler 'object step)])
	(let ([rf1 (reset-role-filler 'object step object)]
	      [rf2 (reset-role-filler 'object step
				      (group-splice '() object step-object))])
	  (list->group (list rf1 rf2)))))))
			       


; (apply-repair <pattern> <mop>) takes the group of steps in the
; repair slot of <mop> and puts them in the recipe in place of the
; step that caused the failure.  Apply-repair first has to find which
; group of the steps in the recipe contined the bad step, then it
; splices in the new steps.

(define apply-repair
  (lambda (pattern mop)
    (let ([repair (get-filler 'repair mop)]
	  [cause  (path-filler '(explanation cause) mop)]
	  [recipe (role-filler 'solution mop)])
      (let ([steps (role-filler 'steps recipe)])
	(reset-role-filler 'steps recipe
           (ormap
	     (lambda (slot)
	       (if (group-member? cause (slot-filler slot))
		   (reset-role-filler (slot-role slot)
                      steps
		      (group-splice (group->list repair)
				    cause (slot-filler slot)))
		   #f))
	     (mop-slots steps)))))))
					

; Generalization functions                              (Listing 7.20 p231)
;--------------------------------------------------------------------------

; (generalize-repair <repair-mop> <role-list>) uses the mapping in
; <repair-mop>  and the slots referred to by <role-list> to form an
; abstraction and its negatios.  These are then used to reorganize the
; solution and its siblings in <repair-mop>.  <role-list> should
; spcify those rolse that are given as input to the problem solver.

(define generalize-repair
  (lambda (repair input-roles)
    (let* ([solution (get-filler 'repaired-solution repair)]
	   [slots (generalize-slots solution input-roles
				    (path-filler '(explanation mapping)
						 repair))]
	   [absts (mop-absts solution)])
      (for-each
       (lambda (slot)
	 (slots->mop
	  (forms->slots
	   `((,(slot-role slot)
	      M-not
	      (object ,(slot-filler slot)))))
	  absts #t))
       slots)
      (slots->mop slots absts #t))))
      

; (generalize-slots <solution-mop> <role-list> <map-group>) returns a
; list of generalize slots, one for each role in role-list.  The
; filers are the corresponding fillers in <solution-mop>, generalized
; according to <map-group>.

(define generalize-slots
  (lambda (mop roles maps)
    (let loop
	([roles roles])
      (if (null? roles)
	  '()
	  (let ([g-slot (generalize-slot (car roles)
					 (role-filler (car roles) mop)
					 maps)])
	    (if g-slot
		(cons g-slot
		      (loop (cdr roles)))
		(loop (cdr roles))))))))

; (generalize-slot <role> <mop> <map-group>) returns a role slot, with
; <mop> generalized according to <map-group>

(define generalize-slot
  (lambda (role mop maps)
    (and mop
	 (let ([abst (generalize-mop mop maps)])
	   (and abst (make-slot role abst))))))

; (generalize-mop <mop> <map-group>) looks in <map-group> for a
; mapping with <mop> in the spec slot.  If it finds one,  it returns
; the corresponding abst filler.

(define generalize-mop
  (lambda (mop maps)
    (ormap
      (lambda (slot)
	(if (eqv? (role-filler 'spec (slot-filler slot)) mop)
	    (role-filler 'abst (slot-filler slot))
	    #f))
      (mop-slots maps))))


; MOP instances for Chef examples                        (Listing 7.21 p232)
;--------------------------------------------------------------------------

; A function that 'explains' the soggy broccoli failure.  It builds an
; instance of a predefined explanation that has the information to
; make the example work.  (It doesn't do any real explanation.)

; (chef-explain <mop>) takes a failure mop and returns an explanation
; for the failure.  The step blamed for the failure is taken from
; *bad-step* which is set by chef-demo.

(define chef-explain
  (lambda (mop)
    (slots->mop
     `(instance
       (failure ,mop)
       (cause ,*bad-step*)
       (rule M-rule)
       (mapping
	,(slots->mop
	  (forms->slots
	   '((1 M-map instance
		(absts M-meat) (spec I-M-beef))
	     (2 M-map instance
		(abst M-crisp-vegetable)
		(spec I-M-broccoli))))
	  '(M-Map-group)
	  #t)))
     '(M-explanation)
     #t)))

       
; Micro Chef test function                             (Listing 7.22 p 233)
;--------------------------------------------------------------------------

; (run-chef <slot-list>) calls chef with the given slots.  If it
; returns a recipe the steps are printed and the recipe is returned.

; Added a check to make sure the input slots actually had mops in them
; [4/7/94 R.S.]

(define run-chef
  (lambda (slots)
    (insist run-chef
	    (andmap (lambda (slot) (mop? (slot-filler slot))) slots))
    (let ([recipe (chef slots)])
      (and recipe
	   (begin
	     (printf "--------------------~n")
	     (printf "The steps in ~a are:~n" recipe)
	     (dph (role-filler 'steps recipe))
	     recipe)))))


; Micro Chef test cases                                 (Listing 7.23 p234)
;--------------------------------------------------------------------------

; finve functions that are called in order to run the chef demo.

; chef1 creates the beef and broccoli recipe.

(define chef1
  (lambda ()
    (set! *bad-recipe*
	  (run-chef '((meat I-M-beef) (vege I-M-broccoli)
		      (taste I-M-hot)
		      (style I-M-stir-fried))))
    (set! *bad-step*
	  (path-filler `(steps stir-fry-steps 1)
		       *bad-recipe*))
    *bad-recipe*))



; chef2 invokes an instance of a canned explanation for the broccoli
; getting soggy because it was cooked with the beef.

(define chef2
  (lambda ()
    (set! *bad-recipe-explanation*
	  (chef-explain 
	   (slots->mop '((state I-M-soggy)
			 (object I-M-broccoli))
		       '(m-failure)
		       #t)))))

; chef3 repairs the recipe created by chef1


(define chef3
  (lambda ()
    (set! *recipe-repair*
	  (chef-repair
	   `((solution ,*bad-recipe*)
	     (explanation ,*bad-recipe-explanation*))))
    (set! *good-recipe*
	  (role-filler 'repaired-solution
		       *recipe-repair*))))




; chef4 uses the repair created by chef3 to reorganize the recipes,
; distinguishing them by whether they contain a meat and crisp
; vegetable or not.

(define chef4
  (lambda ()
    (generalize-repair *recipe-repair*
		       '(meat vege vege2 taste style))))

; chef5 creates a recipe by adapting the repaired recipe.

(define chef5
  (lambda ()
    (run-chef '((meat I-M-chicken) (vege I-M-snow-peas)
		(style I-M-stir-fried)))))

; chef6 which doesn't involve a crips vegetable uses one of the
; original recipes.

(define chef6
  (lambda ()
    (run-chef '((meat I-M-beef) (vege I-M-green-peppers)
		(style I-M-stir-fried)))))

; Micro Chef demo                                       (Listing 7.24 p235)
;--------------------------------------------------------------------------
; Master function to run thru the above test cases

(define chef-demo
  (let* ([seperator (format "~a" (make-string 60 #\~))]
	[print-seperator
	 (lambda ()
	   (printf "~n~a~n" seperator))])
    (lambda ()
      (printf "----------------~n")
      (printf "** Chef1:     Beef and broccoli recipe~n~n")
      (insist chef-demo (chef1))
      (print-seperator)

      (printf "** Chef2:     Kludging soggy broccoli failure~n~n")
      (link-abst *bad-recipe* 'm-failed-solution)
      (printf "Cause is ~s~n" *bad-step*)
      (printf "Kludging explanation~n")
      (insist chef-demo (chef2))
      (printf "Explanation is ~s~n"
	      *bad-recipe-explanation*)
      (print-seperator)
      
      (printf "** Chef3:     Trying to repair ~s~n" *bad-recipe*)
      (insist chef-demo (chef3))
      (printf "Repaired beef and broccoli recipe is ~s~n"
	      *good-recipe*)
      (printf "The repaired steps are:~n")
      (dph (role-filler 'steps *good-recipe*))
      (print-seperator)
      
      (printf "** Chef4:     Generalizing repair~n")
      (insist chef-demo (chef4))
      (printf "New hierarchies:~n")
      (dah 'm-recipe)
      (dah 'm-failed-solution)
      (print-seperator)
      
      (printf "** Chef5:     Chicken and snow-peas recipe~n")
      (insist chef-demo (chef5))
      (print-seperator)
      
      (printf "** Chef6:     Beef and green peppers recipe~n")
      (insist chef-demo (chef6)))))


;  a simple reset function

(define chef-reset
  (lambda ()
    (clear-memory)
    (my-l "microMOPs.mops")
    (my-l "microCHEF.mops")))
