;;; Micro TWEAKER
;;; ----------------------------------------------------------------------
;;; Programmer: Alex Kass, Chris Riesbeck


(require "mops")
(require "applier")
(require "print-utils")


;; scheme port note:
;;  may be more #f vs. '() problems to sort out

;;; Micro Tweaker
;;; ----------------------------------------------------------------------

;;; Deficiency: Can't solve more than one problem per tweak.


(define tweak-expl
  (lambda (expl problems)
  (headline "Micro Tweaker")
  (gather problems
    (lambda (problem) (tweak-problem expl problem))
    append)))

(define tweak-problem
  (lambda (expl problem)
    (gather (retrieve-tweaks problem)
      (lambda (tweak) (apply-tweak tweak expl problem))
      append)))


;;; Retrieving tweaks
;;; ----------------------------------------------------------------------


;;; (RETRIEVE-TWEAKS problem) => list of tweaks
;;;   Returns a list of tweaks appropriate to a problem.

(define retrieve-tweaks
  (lambda (problem)

    (printf "Looking for candidate tweaks in memory for ~s.~n" problem)

    (summarize-results (retrieve-mops (list problem) 'm-tweak)
      "... no tweaks found.~n"
      (lambda (tweaks)
	(print-list "... found ~a~n" "... found:~n    " tweaks)))))



;;; Apply a tweak to an explanation to resolve a problem
;;; ----------------------------------------------------------------


(define apply-tweak
  (lambda (tweak expl problem)
    (let ((fn (eval (<- tweak 'function))))

      (printf "Trying tweak ~s on ~s.~n" (<- tweak 'function) expl)

      (summarize-results (fn expl problem)
	"... tweak failed.~n"
	(lambda (tweak)
	  (printf "... tweak generated: ~s.~n" tweak))))))


;;; Tweakers
;;; ----------------------------------------------------------------

;;; A tweaker takes an explanation and a problem and returns a
;;; (possibly empty) list of tweaked explanations.


;;; Tweaker: Replace an actor with a more appropriate one
;;; ----------------------------------------------------------------

;;; This is a dummy tweak, left as an exercise for the reader

;; scheme port note:
;;  for simplicity returning '() to be consistent with other tweaks

(define replace-actor-search-through-stereotypes
  (lambda (expl problem)
    (printf "  [tweak unimplemented]~n")
    '()))


;;; Tweaker: Replace an event with a causal equivalent
;;; ----------------------------------------------------------------

;;; This is a dummy tweak, left as an exercise for the reader

;; scheme port note:
;;  for simplicity returning '() to be consistent with other tweaks


(define replace-action-search-through-causal-rules
  (lambda (expl problem)
    (printf "  [tweak unimplemented]~n")
    '()))


;;; Tweaker: Replace an event with a scriptal equivalent
;;;----------------------------------------------------------------

;;; Algorithm: Given an action that the Accepter says the actor
;;; doesn't do, find the effects of that action in the XP, then
;;; find a script for the actor that includes the same effects.
;;; 
;;; Example: The XP says "Swale jogged," but the accepter rejects
;;; this action for Swale.  The effect of jogging in the XP was
;;; that Swale ran.  REPLACE-ACTION-SEARCH-THROUGH-STEREOTYPES 
;;; looks for a stereotypical script for Swale that includes running.
;;; It finds the horseracing script, so horseracing replaces jogging
;;; in the tweaked XP and associated explanation.


(define replace-action-search-through-stereotypes
  (lambda (expl problem)
    (let ((actor (<- problem 'observed-value))
	  (label (problem-label problem))
	  (effects (problem-effects expl problem)))
      (if (null? effects)
	  '()
	  (gather (find-scripts-with-effects actor effects)
	    (lambda (script) (replace-expl-event expl label script))
	    cons)))))

(define problem-label
  (lambda (problem)
    (cadr (<- problem 'path))))


;;; Find the effects an event had in an explanation
;;; ----------------------------------------------------------------


(define problem-effects
  (lambda (expl problem)

    (printf "Looking for effects to be accounted for.~n")
  
    (summarize-results (find-consequences expl (problem-label problem))
      "... no effects found.~n"
      (lambda (effects)
	(print-list "... found ~a~n" "... found:~n    " effects)))))


(define find-consequences
  (lambda (expl label)
    (let ((labelled-events (label-member (<- expl 'xp) label)))
      (if (null? (cdr labelled-events))
	  '()
	  (list (labelled-event-event (cadr labelled-events)))))))

;; s.p.n.: still suspicious of things that say (list ...)

;;; Find scripts for actor with desired effects
;;; ----------------------------------------------------------------

(define find-scripts-with-effects
  (lambda (actor effects)

    (printf "Looking for scripts for ~s that include" actor)
    (print-list " ~a~n" ":~n  " effects)

    (summarize-results
      (gather (retrieve-scripts actor)
	(lambda (x) x)
	(lambda (script others)
	  (if (script-includes-effects? script effects)
	      (cons script others)
	      others)))
      "... no scripts found.~n"
      (lambda (scripts)
	(print-list "... found ~a~n" "... found:~n    " scripts)))))


(define retrieve-scripts
  (lambda (actor)
    (retrieve-mops (list actor) 'm-script-event)))
  

(define script-includes-effects?
  (lambda (script effects)
    (andmap (lambda (effect) (script-includes-effect? script effect))
            effects)))

(define script-includes-effect?
  (lambda (script effect)
    (ormap (lambda (event) (abst? effect (labelled-event-event event)))
	   (<- script 'events))))


;;; Generate an explanation by replacing a labelled event in its XP
;;; ----------------------------------------------------------------


(define replace-expl-event
  (lambda (expl label event)
    (apply-xp (replace-xp-event (<- expl 'xp) label event)
              (<- expl 'story))))

;; scheme port note:
;;  seems that when replacing events in the new-xp we should
;;  also substitute new for old events in the causals and contraints
;;
;;  note: original lisp version doesn't even include the constraints (bug?)

(define replace-xp-event
  (lambda (xp label new-event)
    (let* ((old-event (label-event xp label))
	   (new-events (replace-labelled-event
			 (<- xp 'events) label new-event))
	   (new-xp (new-xp
		     `((observed . ,(<- xp 'observed))
		       (expected . ,(<- xp 'expected))
		       (events . ,new-events)
		       (causals . ,(<- xp 'causals))
		       (constraints . ,(<- xp 'constraints))))))

      (printf "Creating a new XP ~s with ~s instead of ~s~n"
	new-xp new-event old-event)

      new-xp)))


;;; Functions to handle labelled event lists
;;; ----------------------------------------------------------------

(define make-labelled-event cons)
(define labelled-event-label car)
(define labelled-event-event cdr)


(define label-event
  (lambda (name label)
    (<- name 'events label)))

(define label-member
  (lambda (name label)
    (let loop ((events (<- name 'events)))
      (cond
	[(null? events) '()]
	[(equal? label (labelled-event-label (car events)))
	 events]
	[else (cons (car events) (loop (cdr events)))]))))

(define replace-labelled-event
  (lambda (labelled-events label new-event)
    (subst (make-labelled-event label new-event)
	   (assoc label labelled-events)
	   labelled-events)))


;;; XP maker
;;; ----------------------------------------------------------------

(define new-xp
  (lambda (args)
    (add-mop (gentemp "m-xp") 'm-xp args)))


;;; End of module
;;; ----------------------------------------------------------------

(provide "tweaker")
