;;; ----------------------------------------------------------------------
;;; Micro RETRIEVER
;;; ----------------------------------------------------------------------
;;; Programmer: Alex Kass, Chris Riesbeck


(require "mops")
(require "applier")
(require "print-utils")
(require "utils")

;;; Explanation retriever
;;; ----------------------------------------------------------------

;;; (RETRIEVE-EXPLS story) => list of explanations
;;;   Finds candidate XP's for story that haven't been tried yet
;;;   and returns the explanations each generates.

(define retrieve-expls
  (lambda (story . problems)

    (headline "Micro XP Retriever")
    (printf "Searching for XP's.~n")

    (map (lambda (xp) (apply-xp xp story))
	 (retrieve-untried-xps story
	   (if (null? problems) '() (car problems))))))
     

;;; XP retriever
;;; ----------------------------------------------------------------

(define retrieve-untried-xps
  (lambda (story problems)
  (summarize-results
    (gather (retrieve-xps story problems)
      (lambda (xp) xp)
      (lambda (xp others)
	(if (already-tried-xp? xp story) others (cons xp others))))
    "... no XP's found.~n"
    (lambda (xps)
      (print-list "... found ~a~n" "... found:~n    " xps)))))


(define retrieve-xps
  (lambda (story problems)
    (retrieve-mops
      (append (story->cues story) (problems->cues problems))
      'm-xp)))


(define already-tried-xp?
  (lambda (xp story)
    (not (null? (retrieve-mops (list xp story) 'm-explanation)))))


;;; Generate retrieval cues from story or explanation problems
;;; ----------------------------------------------------------------

(define story->cues
  (lambda (story)
    (list (<- story 'anomaly 'surprising-fact))))

(define problems->cues
  (lambda (problems)
    (if (null? problems)
	'()
	(summarize-results
	  (gather problems (lambda (x) x)
	    (lambda (problem others)
	      (append (problem->cues problem) others)))
	  "... no cues found."
	  (lambda (ignore) #f)))))

(define problem->cues
  (lambda (problem)

    (printf "Looking for retrieval cues in ~s.~n" problem)

    (let ((expected (<- problem 'expected-value))
	  (observed (<- problem 'observed-value)))
    
      (if (null? expected)
	  '()
	  (begin
	    (printf "... Cue: observed ~s, rather than ~s.~n"
	      observed expected)
	    (list observed))))))


;;; End of module
;;; ----------------------------------------------------------------

(provide "retriever")

