;;; ----------------------------------------------------------------
;;; Print utilities for Micro EXPLAINER
;;; ----------------------------------------------------------------------
;;; Programmer: Chris Riesbeck


(require "mops")
(require "index")


;;; Printing headings
;;; ----------------------------------------------------------------------


;;; (HEADLINE string) => undefined
;;;
;;; Print the string in a highly visible way. Currently, adds
;;; surrounding vertical space and underlining.

(define headline
  (lambda (str)
    (printf "~n~a~n~a~n" str (make-string (string-length str) #\=))))


;;; Printing messages
;;; ----------------------------------------------------------------------

;;; (MAINLINE format-string arg arg ...) => undefined
;;; 
;;; Print the arguments, on a separate line.
;;
;; was:
;; (defun mainline (format-string &rest args)
;;     (unless (null format-string)
;;     (apply #'format t "~&~@?~&" format-string args)))

;;; Summarizing results
;;; ----------------------------------------------------------------------

;;; (SUMMARIZE-RESULTS arg format1 format2) => value of arg
;;; 
;;; If the value of arg is NIL, (format t format1) else
;;; (format t format2 value-of-arg).

;; scheme version is a bit different from the LISP version since we
;; don't have quite so many nifty primitive bells and whistles for
;; formatting.

(define summarize-results
  (lambda (results msg-if-null result-printer)
    (if (null? results)
        (printf msg-if-null)
        (result-printer results))
    results))


;;; Printing MOPs
;;; ----------------------------------------------------------------------


;;; (PRINT-MOP name) => name
;;;   Prints the type, abstractions, slots, and other properties of
;;;   a frame.

(define print-mop
  (lambda (name)
    (print-mop-type (frame-prop name 'type))
    (print-mop-absts (absts-of name))
    (print-mop-slots (slots-of name))
    name))

(define print-mop-type
  (lambda (type)
    (unless (or (not type) (null? type))
      (printf "  ~s" type))))

(define print-mop-absts
  (lambda (absts)
    (unless (or (not absts) (null? absts))
      (printf " isa ~s~n" absts))))

(define print-mop-slots
  (lambda (slots)
    (for-each (lambda (slot) (printf "  ~s: ~s~n" (car slot) (cdr slot)))
      slots)))

;;; Print the index in a readable fashion
;;; ---------------------------------------------------------------- 

(define print-index
  (lambda (index)
    (for-each (lambda (entry)
		(printf "~s => ~s~n"
		  (index-entry-labels entry)
		  (index-entry-items entry)))
      index)))


;; printing lists
;; (should be modified to do pretty-printing or at least prettier printing)

(define print-list
  (lambda (singleton-format multiple-format ls)
    (unless (null? ls)
      (if (null? (cdr ls))
	  (printf singleton-format (car ls))
	  (begin
	    (printf multiple-format)
	    (let loop ((ls ls))
	      (if (null? ls)
		  (printf ".~n")
		  (begin
		    (if (null? (cdr ls))
			(display (car ls))
			(printf "~a, " (car ls)))
		    (loop (cdr ls))))))))))
		    

;;; End of module
;;; ----------------------------------------------------------------

(provide "print-utils")

