;;; ----------------------------------------------------------------
;;; Memory Organization Packages (MOPs)
;;; ----------------------------------------------------------------------
;;; Programmer: Chris Riesbeck

(require "frames")
(require "index")
(require "utils")

;;; DEFMOP and DEFINSTANCE
;;; ----------------------------------------------------------------

(extend-syntax (defmop)
  [(defmop name absts (slot0 filler0) ...)
   (add-mop 'name 'absts '((slot0 . filler0) ...))]
  [(defmop name)
   (add-mop 'name '() '())])

(extend-syntax (definstance)
  [(definstance name absts (slot0 filler0) ...)
   (add-instance 'name 'absts '((slot0 . filler0) ...))]
  [(definstance name absts)
   (add-instance 'name '() '())])

(define add-mop
  (lambda (name absts slots)
    (add-mop-frame name
      (if (symbol? absts) (list absts) absts) slots '((type . mop)))))

(define add-instance
  (lambda (name absts slots)
    (add-mop-frame name
      (if (symbol? absts) (list absts) absts) slots '((type . instance)))))

(define add-mop-frame
  (lambda (name absts slots props)
    (unindex-mop name)
    (index-mop (add-frame name absts slots props))
    name))


;;; The MOP index
;;; ----------------------------------------------------------------

(define *memory-index* (make-index))

(define mop-index-fetch
  (lambda (cues)
    (index-fetch cues *memory-index*)))

(define mop-index-store
  (lambda (labels name)
    (index-store labels name *memory-index*)
    name))

(define mop-index-remove
  (lambda (labels name)
    (index-remove labels name *memory-index*)
    name))

;;; Automatic MOP indexing  
;;; ----------------------------------------------------------------

(define index-mop
  (lambda (name)
    (indexer-map mop-index-store name)))

(define unindex-mop
  (lambda (name)
    (indexer-map mop-index-remove name)))

(define indexer-map
  (lambda (fn name)
    (for-each
      (lambda (abst)
	(for-each
	  (lambda (index)
	    (fn (instantiate-index index name) name))
	  (<- abst 'indices)))
      (absts-of name))))

(define instantiate-index
  (lambda (index name)
    (gather index
      (lambda (path) (instantiate-index-path path name))
      (lambda (val result) (if (null? val) result (cons val result))))))

(define instantiate-index-path
  (lambda (path name)
    (path-filler name (if (pair? path) path (list path)))))

;;; Retrieving mops
;;; ----------------------------------------------------------------

(define retrieve-mops
  (lambda (cues abst)
    (gather (mop-index-fetch cues)
      (lambda (x) x)
      (lambda (mop result)
	(if (abst? abst mop) (cons mop result) result)))))

;;; Clearing MOP memory
;;; ----------------------------------------------------------------

(define clear-mop-memory
  (lambda ()
    (clear-frame-memory)
    (clear-index *memory-index*)))


;;; End of module
;;; ----------------------------------------------------------------

(provide "mops")


