;; draft of a frames package modeled on the frames.lisp stuff from microSWALE

(define-structure (frame name absts slots props))

(require "table")

(define frame-of)   ;; gets value below  (returns #f on failure)
(define add-frame)  ;; gets value below
(define clear-frame-memory
  (lambda ()
    (table-init 1009
      (lambda (lookup insert delete)  ;; delete function ignored for now
	(set! frame-of lookup)
	(set! add-frame
	  (lambda (name absts slots props)
	    (insert name (make-frame name absts slots props))
	    name))))))

;; (COLLECT-ABSTS abst-spec) => list of names
;;   Given either a single abstraction or a list of abstractions,
;;   returns a list of abstractions, with any redundancies removed.

(define collect-absts
  (lambda (abst-spec)
    (remove-redundant-absts
      (if (pair? abst-spec) abst-spec (list abst-spec)))))

;; (COLLECT-SLOTS slots-spec) => list of slots
;;   Given a list of the form (role filler role filler ...)
;;   returns a list of slots, one for each role-filler pair.

(define collect-slots
  (lambda (slots-spec)
    (let loop ((specs slots-spec) (slots '()))
      (if (null? specs)
	  slots
	  (loop (cddr specs)
		(cons (make-slot (car specs) (cadr specs)) slots))))))


;; Getting frame components
;; ----------------------------------------------------------------

(define slots-of
  (lambda (source)
    (if (pair? source)
	source
	(let ((frame (frame-of source)))
	  (if frame (frame-slots frame) '())))))

;; scheme port note:
;;  not sure just what this &aux stuff is in the original LISP:
;;  (defun absts-of (name &aux (frame (frame-of name)))
;;    (and (not (null frame))
;;        (frame-absts frame)))
;;  assume it's like the dot interface, but I haven't gotten any
;;  wrong number of args errors yet...

(define absts-of
  (lambda (name)
    (let ((frame (frame-of name)))
      (if frame (frame-absts frame) '()))))

(define props-of
  (lambda (name)
    (let ((frame (frame-of name)))
      (if frame (frame-props frame) '()))))


;; Using the abstraction hierarchy
;; ----------------------------------------------------------------

;; (ABST? abst spec) => true or false
;;   Return true if abst is spec or an abstraction of spec.
;; (STRICT-ABST? abst spec) => true or false
;;   Return true if abst is an abstraction of spec, but not spec
;;   itself.
;; (REMOVE-REDUNDANT-ABSTS absts) => list of absts
;;    Returns absts, minus duplicates and items which are known
;;    abstractions of other items in the list.


(define abst?
  (lambda (abst spec)
    (or (eq? abst spec)	(strict-abst? abst spec))))

(define strict-abst?
  (lambda (abst spec)
    (ormap (lambda (spec-abst) (abst? abst spec-abst))
	   (absts-of spec))))
     
(define remove-redundant-absts
  (lambda (absts)
    (filter absts (lambda (x y) (not (eq? x y))) strict-abst?)))


;; Getting the filler of a slot in a frame
;; ----------------------------------------------------------------

;; (<- name role role ...) => filler
;;   Return the filler found by tracing the roles from the frame
;;   named through its subcomponents. Fillers may be inherited.

(define <-
  (lambda (name . roles)
    (path-filler name roles)))

(define path-filler
  (lambda (name roles)
    (if (null? roles)
        '()
        (let loop ([roles  (cdr roles)]
                   [result (inherit-filler name (car roles))])
         (if (null? roles)
             (or result '())
	     (let ([tmp (inherit-filler result (car roles))])
		(if tmp
		    (loop (cdr roles) tmp)
		    '())))))))

;; scheme port note:
;;  not sure if the above correctly implements the following
;;  from the original LISP source:
;;  (defun path-filler (name roles)
;;    (loop for role in roles
;;          until (null name)
;;          do (setq name (inherit-filler name role))
;;          finally (return name)))


;; Inheriting slots
;; ----------------------------------------------------------------

;; (INHERIT-FILLER name role) => filler
;;   Return either the explicit filler of role in the frame named,
;;   or the most specific filler of role in the frame's abstractions.

(define inherit-filler
  (lambda (name role)
    (and (not (null? name))
	 (or (role-filler name role)
	     (most-specific-inherited-filler name role)))))

(define most-specific-inherited-filler
  (lambda (name role) 
    (let loop ([absts (absts-of name)] [filler #f])
      (if (null? absts)
	  filler
	  (loop (cdr absts)
		(more-specific-filler name role filler (car absts)))))))
      
(define more-specific-filler
  (lambda (name role filler abst)
    (let ((abst-filler (inherit-filler abst role)))
      (cond
	[(more-specific? abst-filler filler) abst-filler]
	[(more-specific? filler abst-filler) filler]
	[else (error 'more-specific-filler
		"~a in ~a has incompatible fillers: ~a and ~a~n"
		role name filler abst-filler)]))))

(define more-specific?
  (lambda (filler1 filler2)
    (or (not filler2)
	(abst? filler2 filler1))))


;; Explicit slots
;; ----------------------------------------------------------------

(define slot-role car)
(define slot-filler cdr)

(define role-filler
  (lambda (name role)
    (let ([slot (role-slot name role)])
      (and slot (slot-filler slot)))))

(define role-slot
  (lambda (source role)
    (assq role (slots-of source))))

;; scheme port note:
;;  again, not sure if the above correctly renders the following LISP:
;;   (defun role-slot (source role) 
;;     (find role (slots-of source) :key #'slot-role))


;; Frame properties
;; ----------------------------------------------------------------

(define frame-prop
  (lambda (name prop)
    (let ((result (assq prop (props-of name))))
      (and result (cdr result)))))

(define set-frame-prop
  (lambda (name prop val)
    (let* ((frame (frame-of name))
	   (props (or (frame-props frame) '())))
      (let ((result (flassq prop props)))
	(if result
	    (set-car! (cdr result) val)
	    (set-frame-props! frame (cons prop (cons val props))))))))

;; scheme port note:
;;  not real sure about the following:
;;  LISP:
;;  (defsetf frame-prop set-frame-prop)

;; Utils
;; ------------------------------------------------------------------

;; flassq is in table.ss which is required by this module.

;; returns a list of those elements in ls that pass all the tests
;; relative to the other elements of the list (just a way of
;; compacting things like remove-redundant-absts and remove-duplicates
;; into one pass)
;;
;; probably not worth it, but it was fun.

(define filter
  (lambda (ls . tests)
    (if (null? tests)
	(error 'filter "no tests to perform")
	(let loop ((ls ls) (ans '()))
	  (if (null? ls)
	      ans
	      (let ((first (car ls))
		    (rest  (cdr ls)))
		(loop rest
		      (if (andmap (lambda (elt)
				    (andmap (lambda (test) (test first elt))
				      tests))
			    rest)
			  (cons first ans)
			  ans))))))))

;; End of module
;; ----------------------------------------------------------------
  
(provide "frames")





