;;; ----------------------------------------------------------------------
;;; Micro-EXPLAINER
;;; ----------------------------------------------------------------------
;;; Programmer: Chris Riesbeck

;;; Top-level controller for Micro Explainer

(require "mops")
(require "print-utils")
(require "retriever")
(require "accepter")
(require "tweaker")
(require "queue")

;;; The Explainer
;;; ----------------------------------------------------------------------


;;; (EXPLAIN &optional story) => explanation
;;;    Returns an explanation for the story, if any.
;;;    
;;; If called again with no arguments, gets another explanation,
;;; if any.
;;;
;;; Ex.  (EXPLAIN 'M-SWALE-STORY) => first explanation
;;;      (EXPLAIN) => second explanation

(define explain
  (lambda story
    (unless (null? story)
      (initialize-explainer (car story)))
    (run-explainer)))


;;; Initialize the explainer for a new story
;;; ----------------------------------------------------------------

(define *story*)

(define initialize-explainer
  (lambda (story)

    (headline "Micro Explainer")
    (printf "Trying to explain ~s " story)
    (printf "which has the anomaly:~n")
    (print-mop (<- story 'anomaly))
    
    (set! *story* story)
    (initialize-explanation-queue (retrieve-expls story))))


;;; Run the explainer on the current story
;;; ----------------------------------------------------------------

(define run-explainer
  (lambda ()
    (iterate-queue-until acceptable-explanation *expls*)))

(define acceptable-explanation
  (lambda (expl)
    (let ((problems (evaluate-expl expl 'prediction)))
      (case (explanation-status expl problems)
	[(acceptable) expl]
	[(tweakable)
	 (add-explanations (tweak-expl expl problems))
	 (add-explanations (retrieve-expls *story* problems))
	 #f]
	[else #f]))))

;;; The internal explanation queue
;;; ----------------------------------------------------------------

(define *expls* (make-queue))

(define initialize-explanation-queue
  (lambda (expls)
    (if (not (queue-empty? *expls*))
	(set! *expls* (make-queue))
	(add-explanations expls))))

(define explanation-queue (lambda () *expls*))

(define next-explanation (lambda () (dequeue *expls*)))

(define add-explanations
  (lambda (expls)
    (for-each (lambda (expl) (enqueue expl *expls*)) expls)))

;;; End of module
;;; ----------------------------------------------------------------

(provide "explainer")

