;;; ----------------------------------------------------------------
;;; Micro ACCEPTER 
;;; ----------------------------------------------------------------------
;;; Programmer: David Leake, Alex Kass, Chris Riesbeck


(require "mops")
(require "print-utils")

;; scheme port note:
;;  potential source of bugs here is use of append exploiting
;;  LISP equivalence of NIL and #f.  may not have changed all
;;  of these cases.

;;; Explanation evaluation
;;; ----------------------------------------------------------------------
;;; 
;;; (EVALUATE-EXPL explanation) => problems
;;;   Collects problems with the explanation.

(define evaluate-expl
  (lambda (expl purpose)
    (headline "Micro-Accepter")
    (printf "Evaluating ~s for the purpose of ~s.~n" expl purpose)
    (printf "The underling XP is:~n")
    (print-mop (<- expl 'xp))

    (summarize-results (collect-problems expl purpose)
      "Explanation has no problems."
      (lambda (ls)
	(print-list "Problem found: ~a~n" "Problems found:~n  " ls)))))


(define collect-problems
  (lambda (expl purpose)
    ;; force order of evaluation to match that of the lisp system
    ;; so the trace output will match up
    (let* ((relevance-problems      (relevance-problems expl))
	   (believability-problems  (believability-problems expl))
	   (usefulness-problems     (usefulness-problems expl purpose)))
      (append relevance-problems believability-problems usefulness-problems))))


;;; Problem evaluation
;;; ----------------------------------------------------------------
;;; 
;;; Classifies a list of problems as acceptable, tweakable, or
;;; hopeless. 

(define explanation-status
  (lambda (expl problems)
    (cond
      [(null? problems) 'acceptable]
      [else
	(printf "Checking seriousness of problems.~n")

	(summarize-results
	  (reduce worse-problem-status problems 'acceptable)
	  "Couldn't determine status!~n"
	  (lambda (status)
	    (printf "... ~a is ~a~n" expl status)))])))

               
(define worse-problem-status
  (lambda (status problem)
    (let ((problem-status (<- problem 'status)))
      (if (worse-status? problem-status status)
	  problem-status
          status))))

(define worse-status?
  (lambda (status1 status2)
    (and status1
	 (let ((status-scale '(hopeless tweakable acceptable)))
	   (and (memq status2 (cdr (memq status1 status-scale))) #t)))))


;;; Relevance
;;; ----------------------------------------------------------------------

(define relevance-problems
  (lambda (expl)
    (let ((fact (<- expl 'story 'anomaly 'surprising-fact)))

      (printf "Checking if explanation accounts for ~s.~n" fact)

      (summarize-results
	(expl-relevance-problems expl fact)
	"... appears relevant.~n"
	(lambda (x)
	  (printf "... ~a does not appear relevant.~n" expl))))))

(define expl-relevance-problems
  (lambda (expl fact)
    (if (abst? (<- expl 'xp 'events 'outcome) fact)
	'()
         (list
	   (new-problem
	     'm-irrelevant-to-surprising-feature-problem
	     `((explanation . ,expl) (feature-unaccounted-for . ,fact)))))))

;;; Explanation believability
;;; ----------------------------------------------------------------------

(define believability-problems
  (lambda (expl)

    (printf "Checking believability.~n")

    (summarize-results (expl-constraints-problems expl)
      "... no problems.~n"
      (lambda (ignore)
	(printf "... explanation is not believable.~n")))))


;;; Apply XP constraints to the known outcome
;;; ----------------------------------------------------------------

(define expl-constraints-problems
  (lambda (expl)
    (let ((fact (<- expl 'story 'anomaly 'surprising-fact)))
      (gather (<- expl 'xp 'constraints)
	(lambda (constraint) (constraint-problems constraint fact expl))
	append))))


;; seems we only apply the constraint if it has an outcome path
;; if so, this might be more clear if we have a "has-outcome-path?"
;; predicate and only apply the constraint when that yields true

(define constraint-problems
  (lambda (constraint fact expl)
    (gather constraint
      (lambda (x) x)
      (lambda (path others)
	(if (outcome-path? path)
	    (append (apply-constraint constraint fact expl
		      (outcome-path-path path))
	            others)
	    others)))))


;;; Collect the constraint paths referring to the outcome
;;; ----------------------------------------------------------------

(define outcome-paths
  (lambda (constraint)
    (gather constraint
      (lambda (x) x)
      (lambda (path others)
	(if (outcome-path? path)
	    (cons (outcome-path-path path) others)
	    others)))))

(define outcome-path?
  (lambda (path)
    (and (pair? path)
	 (eq? (car path) 'events)
	 (eq? (cadr path) 'outcome))))

(define outcome-path-path cddr)


;;; Apply the constraint to a particular component of fact
;;; ----------------------------------------------------------------

;;; Note: since the fact(s) refer to what was true, and the
;;; outcome refers to what has become true, don't use outcome
;;; related paths as constraints.

(define apply-constraint
  (lambda (constraint fact expl path)
    (gather constraint
      (lambda (x) x)
      (lambda (constrainer others)
	(if (outcome-path? constrainer)
	    others
	    (let ((probs (norm-filler-problems
			   (expl-norm expl constrainer)
			   (fact-filler fact path)
			   constrainer
			   expl)))
	      (if probs (cons probs others) others)))))))

(define expl-norm
  (lambda (expl constrainer)
    (path-filler (<- expl 'xp) constrainer)))

(define fact-filler
  (lambda (fact path)
    (path-filler fact path)))

(define norm-filler-problems
  (lambda (expected observed constrainer expl)
    (if (or (null? expected) (abst? expected observed))
	#f
	(new-problem 'm-non-normative-filler-problem
	  `((explanation . ,expl)
	    (path . ,constrainer)
	    (expected-value . ,expected)
	    (observed-value . ,observed))))))


;;; Usefulness check
;;; ----------------------------------------------------------------------

(define usefulness-problems
  (lambda (expl purpose)

    (printf "Checking if explanation is useful for the purpose of ~s.~n"
      purpose)

    (summarize-results
      (case purpose
	[(accounting-for-event)
	 (printf "... Already tested by relevance check.~n")
	 '()]
	[(prediction) (usefulness-for-prediction-problems expl)])
      (format "... Explanation is adequate for ~s.~n" purpose)
      (lambda (ignore)
	(printf
	  "... Explanation ~a is not adequate for ~s.~n" expl purpose)))))


;;; Check usefulness for prediction
;;; ----------------------------------------------------------------------

(define usefulness-for-prediction-problems
  (lambda (expl)

    (printf "Is the explanation predictive?~n")

    (gather (<- expl 'xp 'events)
      (lambda (item)
	(knowability-problems expl (car item) (cdr item)))
      (lambda (x y) (if x (cons x y) y)))))


(define knowability-problems
  (lambda (expl label fact)
    (and (not (knowable? fact))
	 (new-problem 'm-non-knowable-antecedent-problem
	   `((explanation . ,expl)
	     (fact . ,fact)
	     (event-label . ,label))))))


;;; Knowable?
;;; ----------------------------------------------------------------------

(define knowable?
  (lambda (fact)
    (not (abst? 'm-hidden-state fact))))


;;; Evaluation construction
;;; ----------------------------------------------------------------------

(define new-evaluation
  (lambda args
    (add-instance (gentemp "m-expl-eval") 'm-explanation-evaluation args)))


;;; Problem construction
;;; ----------------------------------------------------------------------


(define new-problem
  (lambda (type args)
    (printf "Problem found!~n")
    (print-mop
      (add-instance (gentemp "m-prob") type args))))


;;; End of module
;;; ----------------------------------------------------------------

(provide "accepter")

