;;; ----------------------------------------------------------------
;;; Index Manager
;;; ----------------------------------------------------------------
;;; Programmer: Chris Riesbeck


(defstruct index entries)

(defstruct (index-entry (:type list)) labels items)


;;; Primary indexing functions
;;; ----------------------------------------------------------------

(defun clear-index (index)
  (setf (index-entries index) nil))

(defun index-fetch (cues index)
  (loop for (labels items) in (index-entries index)
        when (subsetp labels cues :test #'abstp)
         append items))

(defun index-store (labels item index)
  (unless (null labels)
    (let ((entry (index-entry labels index)))
      (cond ((null entry)
             (add-index-entry labels item index))
            (t (add-index-item item entry)))
      item)))

(defun index-remove (labels item index)
  (let ((entry (index-entry labels index)))
    (cond ((null entry) nil)
          ((not (member item (index-entry-items entry))) nil)
          (t (remove-index-item item entry) t))))


;;; Getting the entry for a set of labels (not cues!)
;;; ----------------------------------------------------------------

(defun index-entry (labels index)
  (find labels (index-entries index)
        :test #'set-equalp
        :key #'index-entry-labels))

(defun set-equalp (x y)
  (and (subsetp x y) (subsetp y x)))


;;; Adding and removing items
;;; ----------------------------------------------------------------

(defun add-index-entry (labels item index)
  (push (make-index-entry :labels labels
                          :items (list item)) 
        (index-entries index)))

(defun add-index-item (item entry)
  (pushnew item (index-entry-items entry)))

(defun remove-index-item (item entry)
  (setf (index-entry-items entry)
        (remove item (index-entry-items entry))))


;;; End of module
;;; ----------------------------------------------------------------

(provide "index")

