;;; ----------------------------------------------------------------
;;; Micro Frame System
;;; ----------------------------------------------------------------------
;;; Programmer: Chris Riesbeck



;;; Frame and slot structures
;;; ----------------------------------------------------------------


(defstruct frame name slots absts props)

;;; We implement slots as simple lists to simplify handling fillers
;;; that are labelled lists.

(defstruct (slot (:type list)) role filler)


;;; Creating frames
;;; ----------------------------------------------------------------

(defvar *frames* (make-hash-table))

(defun clear-frame-memory ()
  (clrhash *frames*))

(defun frame-of (name) (gethash name *frames*))

(defun add-frame (&key name absts slots props)
  (setf (gethash name *frames*)
        (make-frame :name name
                    :absts (collect-absts absts)
                    :slots (collect-slots slots)
                    :props props))
  name)


;;; Auxiliary functions:
;;;
;;; (COLLECT-ABSTS abst-spec) => list of names
;;;   Given either a single abstraction or a list of abstractions,
;;;   returns a list of abstractions, with any redundancies removed.

(defun collect-absts (abst-spec)
  (remove-redundant-absts
   (if (listp abst-spec) abst-spec (list abst-spec))))


;;; (COLLECT-SLOTS slots-spec) => list of slots
;;;   Given a list of the form (role filler role filler ...)
;;;   returns a list of slots, one for each role-filler pair.

(defun collect-slots (slots-spec)
  (loop for (role filler) on slots-spec by #'cddr
        collect (make-slot :role role :filler filler)))


;;; Getting frame components
;;; ----------------------------------------------------------------


(defun slots-of (source)
  (if (consp source) 
      source
      (let ((frame (frame-of source)))
        (and (not (null frame)) 
             (frame-slots frame)))))
        
(defun absts-of (name &aux (frame (frame-of name)))
  (and (not (null frame))
       (frame-absts frame)))
        
(defun props-of (name &aux (frame (frame-of name)))
  (and (not (null frame))
       (frame-props frame)))


;;; Using the abstraction hierarchy
;;; ----------------------------------------------------------------

;;; (ABSTP abst spec) => true or false
;;;   Return true if abst is spec or an abstraction of spec.
;;; (STRICT-ABSTP abst spec) => true or false
;;;   Return true if abst is an abstraction of spec, but not spec
;;;   itself.
;;; (REMOVE-REDUNDANT-ABSTS absts) => list of absts
;;;    Returns absts, minus duplicates and items which are known
;;;    abstractions of other items in the list.


(defun abstp (abst spec)
 (or (eql abst spec)
     (strict-abstp abst spec)))


(defun strict-abstp (abst spec)
  (loop for spec-abst in (absts-of spec)
        thereis (abstp abst spec-abst)))


(defun remove-redundant-absts (absts)
  (let ((l (remove-duplicates absts)))
   (set-difference l l :test #'strict-abstp)))


;;; Getting the filler of a slot in a frame
;;; ----------------------------------------------------------------

;;; (<- name role role ...) => filler
;;;   Return the filler found by tracing the roles from the frame
;;;   named through its subcomponents. Fillers may be inherited.


(defun <- (name &rest roles)
  (path-filler name roles))

(defun path-filler (name roles)
  (loop for role in roles
        until (null name)
        do (setq name (inherit-filler name role))
        finally (return name)))


;;; Inheriting slots
;;; ----------------------------------------------------------------


;;; (INHERIT-FILLER name role) => filler
;;;   Return either the explicit filler of role in the frame named,
;;;   or the most specific filler of role in the frame's abstractions.

(defun inherit-filler (name role)
  (or (role-filler name role)
      (most-specific-inherited-filler name role)))

(defun most-specific-inherited-filler (name role)
  (let ((filler nil))
    (dolist (abst (absts-of name))
      (setq filler (more-specific-filler name role filler abst)))
    filler))

(defun more-specific-filler (name role filler abst)
  (let ((abst-filler (inherit-filler abst role)))
    (cond ((more-specific-p abst-filler filler) abst-filler)
          ((more-specific-p filler abst-filler) filler)
          (t (error "~S in ~S has incompatible fillers: ~S and ~S" ;
                    role name filler abst-filler)))))

(defun more-specific-p (filler1 filler2)
  (or (null filler2)
      (abstp filler2 filler1)))


;;; Explicit slots
;;; ----------------------------------------------------------------

    
(defun role-filler (name role) 
  (let ((slot (role-slot name role)))
    (and slot (slot-filler slot))))

(defun role-slot (source role) 
  (find role (slots-of source) :key #'slot-role))



;;; Frame properties
;;; ----------------------------------------------------------------

(defun frame-prop (name prop)
  (getf (props-of name) prop))

(defun set-frame-prop (name prop val)
  (setf (getf (frame-props (frame-of name)) prop) val))

(defsetf frame-prop set-frame-prop)


;;; End of module
;;; ----------------------------------------------------------------
  
(provide "frames")



